﻿﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using VeteransAffairs.Registries.Business;
using VeteransAffairs.Registries.BusinessManager;
using VeteransAffairs.Registries.BusinessManager.TBI;
using BusinessHelper = VeteransAffairs.Registries.Business.Utilities.Helpers;

public partial class TBI_Instruments_FunctionalMobilityAssessment : BaseTBIInstrumentPage {

    #region Variable Declarations

    private const string FailedFormValuesName = "FunctMobilityFailedFormValues";
    private const string DeferredMessageKeyName = "FunctMobilityDeferredMessageKey";
    protected int _surveyTypeId = 15; // Database Id for Functional Mobility Assessment

    PatientManager patientManager = new PatientManager();
    InstrumentType instrumentType = new InstrumentType();

    protected List<int> q1ChoiceIds = new List<int>(); // just for this instrument.

    SURVEY _survey;   // the root object in the Survey set

    List<STD_QUESTION> QuestionList = new List<STD_QUESTION>();
    #endregion

    #region Public Methods

    /// <summary>
    /// Default Constructor
    /// </summary>
    public TBI_Instruments_FunctionalMobilityAssessment() {
        //Manager Save Notification Event Handler
        patientManager.BOSaveSuccess += manager_BOSaveSuccess;
    }

    #endregion Public Methods

    #region Protected Methods

    protected void Page_Init(object sender, EventArgs e)
    {
        VeteransAffairs.Registries.QuestionChoiceRetriever.Reset(_tbiInstrumentManager, _surveyTypeId);
    }

    protected void Page_Unload(object sender, EventArgs e)
    {
        VeteransAffairs.Registries.QuestionChoiceRetriever.Reset(null, -1);
    }

    bool preventNotification = false;
    protected void Page_PreRenderComplete(object sender, EventArgs e)
    {
        Dictionary<string, string> cache;
        if (!Page.IsPostBack && Helpers.SetupPageFromCache(this, FailedFormValuesName, out cache))
        {
            RestoreDateFromCache("D", cache);
            SuppressSaveMessage = false;
            SaveFailMessage = "Unable to save the survey.\r\n\r\nThere may be network issues preventing you from saving.\r\n\r\nPlease try to submit your changes again.";
            RenderStartUpMessage(false);
        }
        else if (Page.IsPostBack && !preventNotification)
        {
            DeferredMessage = "Save was Successful!";
        }
        else if (DeferredMessage != null)
        {
            SuppressSaveMessage = false;
            SaveSuccessMessage = DeferredMessage;
            RenderStartUpMessage(true);
            DeferredMessage = null;
        }
    }

    private void RestoreDateFromCache(string questionNumber, Dictionary<string, string> cache)
    {
        string dateString;
        if (cache.TryGetValue(questionNumber, out dateString))
        {
            dpDate1.SelectedDate = dpDate1.FocusedDate = DateTime.Parse(dateString);
        }
    }

    private static string DeferredMessage
    {
        get { return HttpContext.Current.Session[DeferredMessageKeyName] as string; }
        set { HttpContext.Current.Session[DeferredMessageKeyName] = value; }
    }

    /// <summary>
    /// Page Load method
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void Page_Load(object sender, EventArgs e)
    {
        try
        {
            this.CheckActiveStatus(_surveyTypeId);  // confirm instrument is active - if not, a redirect will occur.

            _tbiInstrumentManager.TbiServiceTo = base.GetTBIServiceTo();
            if (_tbiInstrumentManager.TbiServiceTo == null)
            {
                Logging.WriteLogToFile("FunctionalMobilityAssessment.Page_Load(): _tbiInstrumentManager.TbiServiceTo == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }
            string instTitle = _tbiInstrumentManager.GetInstrumentByID(_surveyTypeId).Name;

            LabelBreadCrumb.Text = "TBI Instruments > Instrument Associations > " + instTitle;
            LabelInstrumentTitle.Text = instTitle;
            Page.Title = "TBI Instruments - " + instTitle;

            Label labelUser = (Label)Master.FindControl("LabelCurrentUser");
            labelUser.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.UserName);

            LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
            LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum);
            if (!IsPostBack)
            {
                base.AddtoViewStateTBIServiceTo(_tbiInstrumentManager.TbiServiceTo);

                if (instrumentType.Status != null)
                {
                    _tbiInstrumentManager.TbiServiceTo.SelectedInstrumentType = instrumentType;
                }

                string _snum = _tbiInstrumentManager.TbiServiceTo.PatientSnum;
                if (!string.IsNullOrEmpty(_snum))
                {
                    string patICN = patientManager.GetPatientICNBySnum(_snum);

                    if (!string.IsNullOrEmpty(patICN))
                    {
                        ViewState["PatientICN"] = patICN;
                        RefreshSurvey(patICN);
                    }
                    else
                    {
                        throw new ArgumentException("Patient with this SSN does not exist.");
                    }

                    if (_survey != null && !_survey.isNew)
                    {
                        SetUpSurvey(_survey);
                    }
                    string referrer = "";
                    if (Request.UrlReferrer != null)
                    {
                        referrer = Request.UrlReferrer.PathAndQuery;
                    }
                    ViewState["referrer"] = referrer;
                }
                LabelPatient.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientName);
                LabelSSN.Text = BusinessHelper.FixStringToPreventXss(_tbiInstrumentManager.TbiServiceTo.PatientSnum.Trim().Length == 9 ? String.Format("{0:000-00-0000}", int.Parse(_tbiInstrumentManager.TbiServiceTo.PatientSnum)) : _tbiInstrumentManager.TbiServiceTo.PatientSnum);
            }
            QuestionList = GetAllQuestionsForSurvey();
            SetQuestionLabels();

            dpDate1.MaxDate = DateTime.Now;
        }
        catch (Exception ex)
        {
            //Redirect to message that session is ended and need to log back in from CPRS.
            Exception _tbilogerror = HandleTBIException(ex, "FunctionalMobilityAssessment.aspx", "100027", "Page_Load()");
            Response.Redirect("Warning.aspx?err=2", false);
        }
    }

    private void SetQuestionLabels()
    {
        LabelDate1.Text = GetQuestionText("D");

        // Set control values for Diagnosis Question
        ctrlDiagnosis.QuestionText = GetQuestionText("DX");
        ctrlDiagnosis.SurveyTypeCode = HiddenSurveyType.Value;

        // Set control values for questions
        ctrlTechDev.QuestionText = GetQuestionText("1");
        ctrlTechDev.SurveyTypeCode = HiddenSurveyType.Value;

        LabelForQ2.Text = GetQuestionText("2");
        LabelForQ2Comment.Text = GetQuestionText("2Comment");
        LabelForQ3.Text = GetQuestionText("3");
        LabelForQ3Comment.Text = GetQuestionText("3Comment");
        LabelForQ4.Text = GetQuestionText("4");
        LabelForQ4Comment.Text = GetQuestionText("4Comment");
        LabelForQ5.Text = GetQuestionText("5");
        LabelForQ5Comment.Text = GetQuestionText("5Comment");
        LabelForQ6.Text = GetQuestionText("6");
        LabelForQ6Comment.Text = GetQuestionText("6Comment");
        LabelForQ7.Text = GetQuestionText("7");
        LabelForQ7Comment.Text = GetQuestionText("7Comment");
        LabelForQ8.Text = GetQuestionText("8");
        LabelForQ8Comment.Text = GetQuestionText("8Comment");
        LabelForQ9.Text = GetQuestionText("9");
        LabelForQ9Comment.Text = GetQuestionText("9Comment");
        LabelForQ10.Text = GetQuestionText("10");
        LabelForQ10Comment.Text = GetQuestionText("10Comment");
        LabelForQ11.Text = GetQuestionText("11");
        LabelForQ11Comment.Text = GetQuestionText("11Comment");
    }

    protected override void LoadViewState(object savedState)
    {
        base.LoadViewState(savedState);
        string surveyXml = (string)ViewState["surveyXml"];
        _survey = SerializerUtil.Deserialize<SURVEY>(surveyXml);
    }

    protected override object SaveViewState()
    {
        ViewState["surveyXml"] = _survey.Serialize();
        return base.SaveViewState();
    }

    /// <summary>
    /// Returns the Question_Text of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected string GetQuestionText(string questionId)
    {
        foreach (STD_QUESTION ques in QuestionList)
        {
            if (ques.QUESTION_NUMBER.ToLower() == questionId.ToLower())
            {
                return ques.QUESTION_TEXT;
            }
        }
        return "";
    }

    /// <summary>
    /// Returns the Question_Id of the question with the passed Question_Number field
    /// </summary>
    /// <param name="questionId"></param>
    /// <returns></returns>
    protected int GetQuestionId(string questionNumber)
    {
        foreach (STD_QUESTION ques in QuestionList)
        {
            if (ques.QUESTION_NUMBER.ToLower() == questionNumber.ToLower())
            {
                return ques.ID;
            }
        }
        return 0;
    }

    /// <summary>
    /// Save Draft button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    public void btnSaveSurveyDraft_Click(object sender, EventArgs e)
    {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        Response.Redirect(Request.Url.PathAndQuery, false);
    }

    /// <summary>
    /// Save and Prepare Note button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnSubmit_Click(object sender, EventArgs e)
    {
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSubmitSurvey.Text + " button again to avoid data loss.";
        Page.Validate();
        SaveCompleteSurvey(ref status);

        if (Page.IsValid && status > 0)
        {
            preventNotification = true;
            Session["survey"] = _survey;
            Response.Redirect("InstrumentSummaryNote.aspx", false);
        }
        else
        {
            Response.Redirect(Request.Url.PathAndQuery, false);
        }
    }

    /// <summary>
    /// Inactive timer firing.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void InActiveTimer_Tick(object sender, EventArgs e)
    {
        preventNotification = true;
        Int32 status = 0;
        SaveFailMessage = "Save Failed. Please press " + btnSaveSurveyDraft.Text + " button again to avoid data loss.";
        SaveCompleteSurvey(ref status);
        Response.Redirect(Request.Url.PathAndQuery, false);
    }

    /// <summary>
    /// Cancel button clicked.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected void btnCancel_Click(object sender, EventArgs e) {
        ViewState["surveyXml"] = null;
        ViewState["PatientICN"] = null;
        ViewState["referrer"] = null;
        Session.Remove("survey");
        Response.Redirect("Instrument.aspx", false);
    }
    #endregion

    #region Private Methods

    /// <summary>
    /// Get all questions for this survey
    /// </summary>
    /// <returns></returns>
    private List<STD_QUESTION> GetAllQuestionsForSurvey() {
        return _tbiInstrumentManager.SelectAllQuestions(_surveyTypeId).ToList();
    }

    /// <summary>
    /// Refresh Survey
    /// </summary>
    /// <param name="patientICN"></param>
    private void RefreshSurvey(string patientICN) {
        var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

        if (surveys.Count > 0) {
            _survey = surveys.First();
        } else {
            _survey = new SURVEY();
        }

        _survey.SetAsChangeTrackingRoot();
    }

    /// <summary>
    /// Save Complete Survey
    /// </summary>
    /// <param name="status"></param>
    private void SaveCompleteSurvey(ref Int32 status)
    {
        try
        {
            string patientICN = String.Empty;
            status = 0;
            //Retrieve patient's ICN from the session variable
            if (ViewState["PatientICN"] != null)
            {
                patientICN = ViewState["PatientICN"].ToString();
                if (String.IsNullOrEmpty(patientICN.Trim()))
                {
                    Logging.WriteLogToFile("FunctionalMobilityAssessment.SaveCompleteSurvey(): WarningRedirect: IsNullOrEmpty(patientICN.Trim()");
                    //Redirect to message that session is ended and need to log back in from CPRS.
                    Response.Redirect("Warning.aspx?err=2", false);
                }
            }
            else
            {
                Logging.WriteLogToFile("FunctionalMobilityAssessment.SaveCompleteSurvey(): WarningRedirect: ViewState[PatientICN] == null");
                //Redirect to message that session is ended and need to log back in from CPRS.
                Response.Redirect("Warning.aspx?err=2");
            }

            if (_survey != null)
            {
                if (_survey.isNew)
                {
                    //Ensure that a concurrent session hasn't already inserted a record for this survey
                    var surveys = _tbiInstrumentManager.SelectByPatientICNAndSurveyType(patientICN, _surveyTypeId, true).ToList();

                    if (surveys.Count > 0)
                    {
                        Logging.WriteLogToFile("FunctionalMobilityAssessment.SaveCompleteSurvey(): WarningRedirect: surveys.Count > 0");
                        //A concurrent session has already inserted a record for this survey and it is not new anymore.
                        Response.Redirect("Warning.aspx?err=9", false);
                    }
                    else
                    {
                        _survey.SURVEY_STATUS = "inprocess";
                        _survey.SetAsInsertOnSubmit();
                        _survey.patient_ICN = patientICN;
                        _survey.STD_SURVEY_TYPE_ID = _surveyTypeId;

                        // TODO: change to datepicker
                        _survey.SURVEY_DATE = DateTime.Today;

                        //Save survey if it's new before adding any responses
                        SuppressSaveMessage = true;
                        _tbiInstrumentManager.Update(_survey);
                        SuppressSaveMessage = false;

                        RefreshSurvey(patientICN);
                    }
                }

                SaveSurvey(ref _survey);
                _survey.SetAsUpdateOnSubmit();
                status = _tbiInstrumentManager.Update(_survey);
                if (status != 0)
                {
                    ((RegistryInstrumentMaster)Master).SetDirtyFlag(false);
                    RefreshSurvey(patientICN);
                    SuppressSaveMessage = false;
                }
            }
        }
        catch (Exception ex)
        {
            Exception _tbilogerror = HandleTBIException(ex, "FunctionalMobilityAssessment.aspx", "100027", "SaveCompleteSurvey()");
        }
        finally
        {
            if (status == 0)
            {
                var cache = Helpers.SavePageToCache(this, FailedFormValuesName);
                ctrlDiagnosis.WriteToCache(cache);
                SaveDateToCache(dpDate1, "D", cache);
            }
            else
                Session[FailedFormValuesName] = null;//clear out the cached version on success
        }
    }

    private static void SaveDateToCache(Telerik.Web.UI.RadDatePicker date, string questionNumber, Dictionary<string, string> cache)
    {
        if (date.SelectedDate != null)
        {
            cache.Add(questionNumber, date.SelectedDate.Value.ToShortDateString());
        }
    }

    private int SelectedDiagnosisID
    {
        get
        {
            string temp = this.ctrlDiagnosis.GetSelectedId();
            if (string.IsNullOrEmpty(temp))
                return 0;
            return Convert.ToInt32(temp);
        }
    }

    private void SaveCustomDiagnosis(SURVEY survey, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        string inputText = ctrlDiagnosis.GetSelectedResultText();
        int selectedID = SelectedDiagnosisID;
        if (null != currentResult)
        {
            if (currentResult.STD_QUESTION_CHOICE_ID != selectedID || currentResult.RESULT_TEXT != inputText)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.RESULT_TEXT = inputText;
                currentResult.STD_QUESTION_CHOICE_ID = selectedID;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else if (selectedID > 0)
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            result.RESULT_TEXT = inputText;
            result.STD_QUESTION_CHOICE_ID = selectedID;
            survey.SURVEY_RESULTs.Add(result);
        }
    }

    private void SaveDate(SURVEY survey, Telerik.Web.UI.RadDatePicker date, string questionNumber)
    {
        string inputText = "";
        if (date.SelectedDate != null)
        {
            inputText = ((DateTime)date.SelectedDate).ToShortDateString();
        }
        SaveText(survey, inputText, questionNumber);
    }

    private void SaveDropDown(SURVEY survey, DropDownList dropDownList, string questionNumber)
    {
        if (dropDownList.SelectedIndex > 0)
        {
            List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
            int onlyChoice = availableChoices.FirstOrDefault().STD_QUESTION_CHOICE_ID;  //There is only one choice for each question
            SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => item.STD_QUESTION_CHOICE_ID == onlyChoice);
            if (null != currentResult)
            {
                if (currentResult.RESULT_TEXT != dropDownList.SelectedValue)
                {
                    currentResult.SetAsUpdateOnSubmit();
                    currentResult.STD_QUESTION_CHOICE_ID = onlyChoice;
                    currentResult.RESULT_TEXT = dropDownList.SelectedValue;
                }
                else
                {
                    currentResult.SetAsNoChangeOnSubmit();
                }
            }
            else
            {
                SURVEY_RESULT result = new SURVEY_RESULT();
                survey.SURVEY_RESULTs.Add(result);
                result.STD_QUESTION_CHOICE_ID = onlyChoice;
                result.RESULT_TEXT = dropDownList.SelectedValue;
            }
        }
    }

    private void SaveSingleSelect(SURVEY survey, RadioButtonList radioButtonList, string questionNumber)
    {
        if (radioButtonList.SelectedIndex > -1)
        {
            List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
            SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
            int selectedValue = Convert.ToInt32(radioButtonList.SelectedValue);
            if (null != currentResult)
            {
                if (currentResult.STD_QUESTION_CHOICE_ID != selectedValue || currentResult.RESULT_TEXT != radioButtonList.SelectedItem.Text)
                {
                    currentResult.SetAsUpdateOnSubmit();
                    currentResult.STD_QUESTION_CHOICE_ID = selectedValue;
                    currentResult.RESULT_TEXT = radioButtonList.SelectedItem.Text;
                }
                else
                {
                    currentResult.SetAsNoChangeOnSubmit();
                }
            }
            else
            {
                SURVEY_RESULT result = new SURVEY_RESULT();
                survey.SURVEY_RESULTs.Add(result);
                result.STD_QUESTION_CHOICE_ID = selectedValue;
                result.RESULT_TEXT = radioButtonList.SelectedItem.Text;
            }
        }
    }

    private void SaveTechDev(SURVEY survey, ASP.controls_techdeviceselector_ascx techDev, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        string selectedText = techDev.GetSelectedResultText();
        if (null != currentResult)
        {
            if (currentResult.RESULT_TEXT != selectedText)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.STD_QUESTION_CHOICE_ID = Convert.ToInt32(techDev.GetSelectedId());
                currentResult.RESULT_TEXT = selectedText;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            survey.SURVEY_RESULTs.Add(result);
            result.STD_QUESTION_CHOICE_ID = Convert.ToInt32(techDev.GetSelectedId());
            result.RESULT_TEXT = selectedText;
        }
    }

    private void SaveText(SURVEY survey, string text, string questionNumber)
    {
        List<STD_QUESTION_CHOICE> availableChoices = VeteransAffairs.Registries.QuestionChoiceRetriever.GetQuestionChoices(questionNumber);
        SURVEY_RESULT currentResult = survey.SURVEY_RESULTs.FirstOrDefault(item => availableChoices.FirstOrDefault(choice => choice.STD_QUESTION_CHOICE_ID == item.STD_QUESTION_CHOICE_ID) != null);
        if (null != currentResult)
        {
            if (currentResult.RESULT_TEXT != text)
            {
                currentResult.SetAsUpdateOnSubmit();
                currentResult.RESULT_TEXT = text;
            }
            else
            {
                currentResult.SetAsNoChangeOnSubmit();
            }
        }
        else if (text != string.Empty)
        {
            SURVEY_RESULT result = new SURVEY_RESULT();
            result.RESULT_TEXT = text;
            result.STD_QUESTION_CHOICE_ID = availableChoices.FirstOrDefault().STD_QUESTION_CHOICE_ID;
            survey.SURVEY_RESULTs.Add(result);
        }
    }

    /// <summary>
    /// Save survey.
    /// </summary>
    /// <param name="survey"></param>
    private void SaveSurvey(ref SURVEY survey) {
        string questionNumber = "";

        try 
        {
            //first, mark all current answers for deletion
            survey.SetAsChangeTrackingRoot();
            foreach (SURVEY_RESULT tempResult in survey.SURVEY_RESULTs)
                tempResult.SetAsDeleteOnSubmit();

            SaveCustomDiagnosis(survey, questionNumber = "DX");
            SaveDate(survey, dpDate1, questionNumber = "D");
            SaveTechDev(survey, ctrlTechDev, questionNumber = "1");
            SaveSingleSelect(survey, rblQ2, questionNumber = "2");
            SaveText(survey, txtCommentsQ2.Text, questionNumber = "2Comment");
            SaveDropDown(survey, ddlPriority1, questionNumber = "2R");
            SaveSingleSelect(survey, rblQ3, questionNumber = "3");
            SaveText(survey, txtCommentsQ3.Text, questionNumber = "3Comment");
            SaveDropDown(survey, ddlPriority2, questionNumber = "3R");
            SaveSingleSelect(survey, rblQ4, questionNumber = "4");
            SaveText(survey, txtCommentsQ4.Text, questionNumber = "4Comment");
            SaveDropDown(survey, ddlPriority3, questionNumber = "4R");
            SaveSingleSelect(survey, rblQ5, questionNumber = "5");
            SaveText(survey, txtCommentsQ5.Text, questionNumber = "5Comment");
            SaveDropDown(survey, ddlPriority4, questionNumber = "5R");
            SaveSingleSelect(survey, rblQ6, questionNumber = "6");
            SaveText(survey, txtCommentsQ6.Text, questionNumber = "6Comment");
            SaveDropDown(survey, ddlPriority5, questionNumber = "6R");
            SaveSingleSelect(survey, rblQ7, questionNumber = "7");
            SaveText(survey, txtCommentsQ7.Text, questionNumber = "7Comment");
            SaveDropDown(survey, ddlPriority6, questionNumber = "7R");
            SaveSingleSelect(survey, rblQ8, questionNumber = "8");
            SaveText(survey, txtCommentsQ8.Text, questionNumber = "8Comment");
            SaveDropDown(survey, ddlPriority7, questionNumber = "8R");
            SaveSingleSelect(survey, rblQ9, questionNumber = "9");
            SaveText(survey, txtCommentsQ9.Text, questionNumber = "9Comment");
            SaveDropDown(survey, ddlPriority8, questionNumber = "9R");
            SaveSingleSelect(survey, rblQ10, questionNumber = "10");
            SaveText(survey, txtCommentsQ10.Text, questionNumber = "10Comment");
            SaveDropDown(survey, ddlPriority9, questionNumber = "10R");
            SaveSingleSelect(survey, rblQ11, questionNumber = "11");
            SaveText(survey, txtCommentsQ11.Text, questionNumber = "11Comment");
            SaveDropDown(survey, ddlPriority10, questionNumber = "11R");
        } 
        catch (Exception surveyEx) 
        {
            Exception _tbilogerror = HandleTBIException(surveyEx, "FunctionalMobilityAssessment.aspx", "100027", "SaveSurvey() - Question " + questionNumber);
            throw;
        }
    }// END SaveSurvey

    private void SetCustomDiagnosisElement(SURVEY_RESULT result, ASP.controls_tbidiagnosisselector_ascx diagControl)
    {
        if (result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
        {
            diagControl.SelectedChoiceId = result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
            if (result.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other"))
            {
                diagControl.SetOtherDescriptionText(result.RESULT_TEXT);
            }
        }
        else
        {
            diagControl.SelectedChoiceId = null;
        }
    }

    private void SetDateElement(SURVEY_RESULT result, Telerik.Web.UI.RadDatePicker date)
    {
        if (!String.IsNullOrEmpty(result.RESULT_TEXT))
        {
            DateTime temp;
            if (DateTime.TryParse(result.RESULT_TEXT, out temp))
            {
                date.SelectedDate = temp;
            }
        }
    }

    private void SetDropDownListElement(SURVEY_RESULT result, DropDownList dropDownList)
    {
        if (!String.IsNullOrEmpty(result.RESULT_TEXT))
        {
            dropDownList.SelectedValue = result.RESULT_TEXT;
        }
    }

    private void SetRadioElement(SURVEY_RESULT result, RadioButtonList radioButton)
    {
        if (result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
        {
            radioButton.SelectedValue = result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();
        }
        else
        {
            radioButton.SelectedIndex = -1;
        }
    }

    private void SetTechDevElement(SURVEY_RESULT result, ASP.controls_techdeviceselector_ascx techDev)
    {
        if (result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID > -1)
        {
            techDev.SelectedChoiceId = result.STD_QUESTION_CHOICE.STD_QUESTION_CHOICE_ID.ToString();

            if (result.STD_QUESTION_CHOICE.CHOICE_TEXT.ToLower().Contains("other"))
            {
                techDev.SetOtherDescriptionText(result.RESULT_TEXT);
            }
        }
        else
        {
            techDev.SelectedChoiceId = null;
        }
    }

    private void SetTextBoxElement(SURVEY_RESULT result, TextBox textBox)
    {
        if (!String.IsNullOrEmpty(result.RESULT_TEXT))
        {
            textBox.Text = result.RESULT_TEXT;
        }
    }

    /// <summary>
    /// Initialize/setup controls
    /// </summary>
    /// <param name="survey"></param>
    private void SetUpSurvey(SURVEY survey)
    {
        try
        {
            foreach (var sResult in survey.SURVEY_RESULTs)
            {
                switch (sResult.STD_QUESTION_CHOICE.STD_QUESTION.QUESTION_NUMBER)
                {
                    case "DX": SetCustomDiagnosisElement(sResult, ctrlDiagnosis); break;
                    case "D": SetDateElement(sResult, dpDate1); break;
                    case "1": SetTechDevElement(sResult, ctrlTechDev); break;
                    case "2": SetRadioElement(sResult, rblQ2); break;
                    case "2Comment": SetTextBoxElement(sResult, txtCommentsQ2); break;
                    case "2R": SetDropDownListElement(sResult, ddlPriority1); break;
                    case "3": SetRadioElement(sResult, rblQ3); break;
                    case "3Comment": SetTextBoxElement(sResult, txtCommentsQ3); break;
                    case "3R": SetDropDownListElement(sResult, ddlPriority2); break;
                    case "4": SetRadioElement(sResult, rblQ4); break;
                    case "4Comment": SetTextBoxElement(sResult, txtCommentsQ4); break;
                    case "4R": SetDropDownListElement(sResult, ddlPriority3); break;
                    case "5": SetRadioElement(sResult, rblQ5); break;
                    case "5Comment": SetTextBoxElement(sResult, txtCommentsQ5); break;
                    case "5R": SetDropDownListElement(sResult, ddlPriority4); break;
                    case "6": SetRadioElement(sResult, rblQ6); break;
                    case "6Comment": SetTextBoxElement(sResult, txtCommentsQ6); break;
                    case "6R": SetDropDownListElement(sResult, ddlPriority5); break;
                    case "7": SetRadioElement(sResult, rblQ7); break;
                    case "7Comment": SetTextBoxElement(sResult, txtCommentsQ7); break;
                    case "7R": SetDropDownListElement(sResult, ddlPriority6); break;
                    case "8": SetRadioElement(sResult, rblQ8); break;
                    case "8Comment": SetTextBoxElement(sResult, txtCommentsQ8); break;
                    case "8R": SetDropDownListElement(sResult, ddlPriority7); break;
                    case "9": SetRadioElement(sResult, rblQ9); break;
                    case "9Comment": SetTextBoxElement(sResult, txtCommentsQ9); break;
                    case "9R": SetDropDownListElement(sResult, ddlPriority8); break;
                    case "10": SetRadioElement(sResult, rblQ10); break;
                    case "10Comment": SetTextBoxElement(sResult, txtCommentsQ10); break;
                    case "10R": SetDropDownListElement(sResult, ddlPriority9); break;
                    case "11": SetRadioElement(sResult, rblQ11); break;
                    case "11Comment": SetTextBoxElement(sResult, txtCommentsQ11); break;
                    case "11R": SetDropDownListElement(sResult, ddlPriority10); break;
                    default: break;
                }
            }
        }
        catch (Exception ex)
        {
            Exception _tbilogerror = HandleTBIException(ex, "FunctionalMobilityAssessment.aspx", "100027", "SetUpSurvey()");
        }
    }// END SetUpSurvey

    #endregion
}